/****************************************************************************
    Copyright (C) 1987-2001 by Jeffery P. Hansen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Last edit by hansen on Tue Aug 29 08:50:01 2000
****************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "tkgate.h"

#define DEBUG_NET 0

int ycIsKW(char*);

static GNet *net_freelist = 0;

int trimName(char *buf)
{
  char *p;
  char *q;
  int did_trim = 0;

  if (isdigit((int)*buf)) {
    memmove(buf+1,buf,strlen(buf)+1);
    *buf = 'D';
    did_trim = 1;
  }

  p = q = buf;

  for (;*q;q++) {
    if (isalpha((int)*q) || isdigit((int)*q) || *q == '_')
      *p++ = *q;
    else if (*q == ' ') {
      *p++ = (p == buf) ? 'S' : '_';
    } else
      did_trim = 1;
  }
  *p = 0;

  return did_trim;
}

void pickValidName(char *buf,char *name,char *base,SHash *H)
{
  char *p;
  int reason = 0;

  if (!base || !*base)
    base = "n";

  if (name) {
    strcpy(buf,name);

    if (trimName(buf)) {
      reason = 1;
      if (!*buf) {
	if (base)
	  strcpy(buf,base);
	else
	  strcpy(buf,"name");
      }
    }

    if (!*buf)
      sprintf(buf,"%s0",base);

    if (H && SHash_find(H,buf))
      reason = 2;
    else if (gateinfo_lookup(buf))
      reason = 3;
    else if (ycIsKW(buf))
      reason = 4;
    else {
      if (reason)
	message(0,msgLookup("err.badnetname"));		/* "Illegal characters in identifier deleted." */
      return;
    }
  } else
    sprintf(buf,"%s0",base);

  p = buf+strlen(buf);

  if ((H && SHash_find(H,buf)) || gateinfo_lookup(buf) || ycIsKW(buf)) {
    char *p;
    int n;

    p = buf + strlen(buf);
    while (p > buf && isdigit((int)p[-1])) p--;

    n = 0;
    if (*p) sscanf(p,"%d",&n);

    do {
      sprintf(p,"%d",n++);
    } while ((H && SHash_find(H,buf)) || gateinfo_lookup(buf) || ycIsKW(buf));
  }

  switch (reason) {
  case 1 :
    message(0,msgLookup("err.netbcrename"),buf);	/* "Identifier renamed to '%s' because of illegal characters." */
    break;
  case 2 :
    message(0,msgLookup("err.netconfnet"),buf);
    break;
  case 3 :
    message(0,msgLookup("err.netconfgat"));
    break;
  case 4 :
    message(0,msgLookup("err.netconfkw"));
    break;
  }

}

GNet *net_newCompatable(char *name,GNet *rnet,GModuleDef *mdef)
{
  GNet *net;
  
  if (net_freelist) {
    net = net_freelist;
    net_freelist = net->next;
  } else
    net = (GNet*) malloc(sizeof(GNet));

  if (mdef) {
    char buf[STRMAX];

    pickValidName(buf,name,"w",&mdef->nets);
    net->signame = strdup(buf);
    if (name) {
      net->show_name = 1;
    } else {
      net->show_name = 0;
    }
  } else {
    net->signame = 0;
    net->show_name = 0;
  }

  net->nbits = rnet ? rnet->nbits : 1;
  net->dtype = rnet ? rnet->dtype : 0;
  net->refs = 0;
  net->mark = 0;
  net->ionet = 0;
  net->mod = mdef;
  net->driver = 0;
  net->decoration = 0;
  net->next = 0;
 
  SHash_insert(&mdef->nets,net->signame,net);

  if (mdef && XGate.es && mdef == XGate.es->env)
    DoTcl("tkg_netListInsert \"%s\" %d %d",net->signame,net->show_name,net->nbits);

#if DEBUG_NET
  printf("net_new() -> %x",net);
  if (net->signame)
    printf("  (%s)\n",net->signame);
  else
    printf("\n");
#endif

  return net;
}

GNet *net_new(char *name,GModuleDef *mdef)
{
  return net_newCompatable(name,0,mdef);
}

void net_update(GNet *n)
{
  DoTcl("tkg_netListUpdate %s %d %d",n->signame,n->show_name,n->nbits);
}

void net_delete(GNet *net)
{
#if DEBUG_NET
  printf("GNet::Delete(%s) %x",net->signame ?: "",net);
#endif

  if (net->signame) {
    SHash_remove(&net->mod->nets,net->signame);
    if (net->mod && XGate.es && net->mod == XGate.es->env)
      DoTcl("tkg_netListRemove %s",net->signame);
    free(net->signame);
#if DEBUG_NET
    printf(" [%d]",r);
#endif
  }
#if DEBUG_NET
  printf("\n");
#endif
  memset(net,0,sizeof(*net));
  net->next = net_freelist;
  net_freelist = net;
}

void net_incref(GNet *net)
{
  net->refs++;
}

void net_decref(GNet *net)
{
  if (--net->refs <= 0)
    net_delete(net);
}

/*
   Select one of the nets 'a' or 'b'
*/
GNet *net_pickOne(GNet *a,GNet *b,int decref_loser)
{
  GNet *r = 0;	/* Net to return */
  GNet *l = 0;	/* Losing net */

  if (!a) 
    r = b;
  else if (!b) 
    r = a;
  else if (a == b) {
    r = a;
    l = b;
  } else if (a->ionet) {
    r = a;
    l = b;
  } else if (b->ionet) {
    r = b;
    l = a;
  } else if (a->show_name) {
    r = a;
    l = b;
  } else if (b->show_name) {
    r = b;
    l = a;
  } else {
    r = a;
    l = b;
  }

  if (l && decref_loser) net_decref(l);

  return r;
}

/*
   Check to see if it is OK to connect wires from the
   specified nets.  Return 1 if OK, return 0 and display
   a status bar message if not OK. 
*/
int net_connectOK(GNet *n1,GNet *n2,int isMidWire)
{
  if (n1 == n2) {
    message(0,msgLookup("err.badconsame"));	/* Connection refused because wires are part of the same net. */
    return 0;
  }
  if (n1->ionet && n2->ionet) {
    message(0,msgLookup("err.badconptsp"));	/* Connection refused because both wires are module ports or supply. */
    return 0;
  }

  if (!isMidWire && n1->nbits != n2->nbits) {
    message(0,msgLookup("err.badconbitw"));	/* Connection refused because bit widths do not match. */
    return 0;
  }

  return 1;
}

/*
  Unselect all nets.
 */
void net_unselect(int drawp)
{
  if (XGate.nsel) {
    GNet *n = XGate.nsel;
    if (drawp) net_draw(n);
    XGate.nsel = 0;
    if (drawp) net_draw(n);
  }
}

/*
  Make 'n' the selected net.
 */
void net_select(GNet *n,int drawp)
{
  if (XGate.nsel == n) return;
  net_unselect(drawp);

  if (!n) return;

  if (drawp) net_draw(n);
  XGate.nsel = n;
  if (drawp) net_draw(n);
}



/*
   Redraw an entire net.
*/
void net_draw(GNet *net)
{
  wire_drawnet(net->driver);
}

void net_setSize(GNet *net,int nbits)
{
  if (net->nbits == nbits) return;

  net->nbits = nbits;

  if (net->mod && XGate.es && net->mod == XGate.es->env) {
    DoTcl("tkg_netListRemove %s",net->signame);
    DoTcl("tkg_netListInsert %s %d %d",net->signame,net->show_name,net->nbits);
  }
}

int listNets(GModuleDef *mdef)
{
  HashElem *nl;

  DoTcl("tkg_clearNetList");
  for (nl = Hash_first(&mdef->nets);nl;nl = Hash_next(&mdef->nets,nl)) {
    GNet *n = (GNet*) HashElem_obj(nl);
    DoTcl("tkg_netListAdd %s %d %d",n->signame,n->show_name,n->nbits);
  }
  DoTcl("tkg_netListEnd");

  return TCL_OK;
}

void net_editProps(GNet *n,int x,int y)
{
  char *sn = n->show_name ? "" :"@";
  int ioCode = 0;
  int wx = x-50+XGate.org_x;
  int wy = y-50+XGate.org_y;

  if (n->ionet) {
    ioCode = n->ionet->typeinfo->Code;
    if (ioCode != LOGICIN && ioCode != LOGICOUT && ioCode != LOGICTRI)
      ioCode = 0;	/* yappari yameta */
  }

  if (n->nbits == 1)
    DoTcl("tkg_editNet %d %d {%s%s} %d\n",wx,wy,n->signame,sn,ioCode);
  else
    DoTcl("tkg_editNet %d %d {%s[%d:0]%s} %d\n",wx,wy,n->signame,n->nbits-1,sn,ioCode);

  if (XGate.tcl->result && strcmp(XGate.tcl->result,"1") == 0) {
    GWire *w = wire_findClosest(n->driver,x,y);
    n->decoration = w->nidx;
  }
}

/*
   Set the signal name of a wire to s.  
*/
void net_rename(GNet *net,char *s,int showName)
{
  char *oldName = strdup(net->signame);
  char buf[STRMAX];

  if (net->signame) {
    DoTcl("tkg_netListRemove %s",net->signame);
    SHash_remove(&net->mod->nets,net->signame);
    free(net->signame);
  }
  net->signame = 0;
  net->show_name = showName;

  pickValidName(buf,s,"w",&net->mod->nets);
  net->signame = strdup(buf);

  SHash_insert(&net->mod->nets,net->signame,net);

  if (net->mod && XGate.es && net->mod == XGate.es->env)
    DoTcl("tkg_netListInsert %s %d %d",net->signame,net->show_name,net->nbits);
  free(oldName);
}

